/*
   Setup:
     1)  0 V low, 4-5 V high TTL signal to ExTrig input (not necessary if
	 using -I or -O options) and all input channels to be tested.  Also,
	 Tach chan 2, if -O option being used (-O2).
     2)  frequency of square wave around 200 Hz maximum, lower if decimation
         factor > 20 or so.
 */

#include <stdlib.h>             /* For exit */
#include <stdio.h>              /* For printf */
#include <string.h>             /* For strcmp, strcat */
#include "e1432.h"
#include "xplot.h"

#define SIZE 16384
#define MAX_LAS		8
#define MAX_CHAN	(MAX_LAS * E1432_CHANS)

#define MAXLONG		0x7fffffffL

/* Wrap this around all the many function calls which might fail */
#define DEBUG(s)        s
#ifdef  __lint
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (func)
#else
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (0)
#endif

#define WIDTH           500
#define HEIGHT          400

#define VALID_OPTS	"b:c:d:e:f:hi:m:n:p:r:s:t:uvw:yA:C:D:I:L:O:PR:T:_"

void
print_usage(char *name)
{
    (void) fprintf(stderr,"%s usage:\n", name);
    (void) fprintf(stderr,
        "    -b blocksize\n"
        "    -c channel (default all) (may appear more than once)\n"
        "    -d decimation factor\n"
        "    -e error tolerance\n"
        "    -f sample frequency\n"
        "    -h (to use trailer trig crtn)\n"
        "    -i measurements_to_ignore\n"
        "    -m blocs per meas restart\n"
        "    -n number of blocks\n"
        "    -p channel (to plot)\n"
        "    -r channel (to compare to)\n"
        "    -s {16 32 32s 32f} (data size)\n"
        "    -t channel (to test)\n"
        "    -v (verbose)\n"
        "    -y (oversampled data)\n"
        "    -A coupleFreq (AC couple)\n"
        "    -C channel (source of trigger correction)\n"
	"       (default or negative - same chan, 0 - no trigger correction)\n"
        "    -D data_mode (b, o, or c)\n"
        "    -L logical address (default 8) (may appear more than once)\n"
        "    -I channel (input channel to trigger from) (default ext)\n"
        "    -O channel (tachometer channel to trigger from) (default ext)\n"
        "    -P (pause during startup, measurments)\n"
        "    -R offset (in samples, of reference)\n"
        "    -T trigger_delay (default -blocksize/2\n"
    );
}

void
parm_err(char *arg, char *parm_name)
{
    (void) fprintf(stderr, "parameter error: %s \"%s\" not converted\n",
      parm_name, arg);
    exit(2);
}


SHORTSIZ16
get_short(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    int rtn = sscanf(arg, "%f", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    if ( ftmp >= 0.0 ) return (SHORTSIZ16)(ftmp + .5);
    else return (SHORTSIZ16)(ftmp - .5);
}


LONGSIZ32
get_long(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    int rtn = sscanf(arg, "%f", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    if ( ftmp >= 0.0 ) return (LONGSIZ32)(ftmp + .5);
    else return (SHORTSIZ16)(ftmp - .5);
}


FLOATSIZ32
get_float(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    int rtn = sscanf(optarg, "%f", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    return ftmp;
}


SHORTSIZ16
get_enum(char *arg, SHORTSIZ16 *enums, char **strs, char *parm_name)
{
    while ( *strs != NULL )
    {
        if ( strcmp(arg, *strs++) == 0 ) return *enums;
	enums++;
    }
    parm_err(arg, parm_name);
}


void prompt(char* promptstr)
{
    char tmp_str[1024];
    (void) printf(promptstr);
    (void) gets(tmp_str);
}

double find_dly(double threshold, FLOATSIZ32 buffer[], LONGSIZ32 blocksize)
{
    double delay;
    int i;

    for ( i = 0; i < blocksize && buffer[i] < threshold; i++);
    delay = (threshold - buffer[i-1]) / ( buffer[i] - buffer[i-1]);
    delay += i - 1;

    return delay;
}


double find_threshold(double rough_threshold, FLOATSIZ32 buffer[],
  LONGSIZ32 blocksize)
{
    double tmp_dly, threshold, pre = 0, post = 0, norm = 0, wt;
    int i, fps, lps;

    /* make a first pass and the pick the pre and post intervals to average */
    tmp_dly = find_dly(rough_threshold, buffer, blocksize);
    fps = tmp_dly - 5;
    lps = tmp_dly + 10;
    if ( fps < 0 || lps >= blocksize ) return rough_threshold;
    for ( i = 0; i < fps; i++ ) pre += buffer[i];
    pre /= (double)fps;
    /* lots of ringing afterwards so do a triangular weighted average */
    for ( i = lps; i < blocksize; i++ )
    {
        wt = (double)(i - lps)/(double)(blocksize-lps);
        norm += wt;
        post += wt * buffer[i];
    }
    post /= norm;
    threshold = (pre + post)/2.0;
    return threshold;
}


int
main(int argn, char **argv)
{
    int opt;               /* really char returned */

    SHORTSIZ16 modules = 1;
    SHORTSIZ16 la_override = 0;
    LONGSIZ32 meas_restart = 0;
    SHORTSIZ16 plot_chan;
    int i, j, status;
    SHORTSIZ16 in_chans = 0;
    SHORTSIZ16 tach_chans = 0;
    SHORTSIZ16 meas_chans, chan;
    struct e1432_hwconfig hwconfig[MAX_LAS];
    FLOATSIZ32 buffer[SIZE];
    FLOATSIZ32 data[2*SIZE];
    LONGSIZ32 count;
    SHORTSIZ16 laddr[MAX_LAS] = { 8 };	/* default logical address */
    SHORTSIZ16 in_chan_list[MAX_CHAN];
    SHORTSIZ16 meas_chan_list[MAX_CHAN];
    SHORTSIZ16 in_group;
    SHORTSIZ16 meas_group;
    E1432ID hw;

    int triggered = 1;  /* tach trig if tach, ext otherwise */
    int plot = 0;
    long N = -1;  /* times through the data acq loop, -1 = infinite */
    LONGSIZ32 blocksize = 64;

    FLOATSIZ32 span;
    FLOATSIZ32 clock_freq = 51200;
    int m, data_offset;
    char* plotid = NULL;
    FLOATSIZ32 fixed_delay;
    FLOATSIZ32 delay_crtn[MAX_CHAN];
    int	tdr_crtn[MAX_CHAN];
    int prev_tdr_crtn[MAX_CHAN];
    FLOATSIZ32 subfs_crtn[MAX_CHAN];
    FLOATSIZ32 prev_subfs_crtn[MAX_CHAN];
    FLOATSIZ32 ref_offset = 0.0;
    LONGSIZ32 dec_factor = 1.0;
    char geometry[80];
    char title[80];
    int row = 0, col = 0;
    int no = 0;
    /*int yes = 1;*/
    int verbose = 0;
    SHORTSIZ16 test_chan = -1;
    SHORTSIZ16 ref_chan = -1;
    SHORTSIZ16 crtn_chan = -1;  /* use each chan's correction */
    int i_test = 0;
    int i_ref = 0;
    SHORTSIZ16 use_trailer = 0;
    SHORTSIZ16 debug_pause = 0;
    SHORTSIZ16 ignore = 0;
    SHORTSIZ16 specic_chans = 0;
    SHORTSIZ16 spec_chan_list[MAX_CHAN];
    SHORTSIZ16 decimation_oversample = E1432_DECIMATION_OVERSAMPLE_OFF;
    SHORTSIZ16 data_size = E1432_DATA_SIZE_32_SERV;
    SHORTSIZ16 data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    SHORTSIZ16 coupling = E1432_COUPLING_DC;
    FLOATSIZ32 coupling_freq = 1.0;
    FLOATSIZ32 range = 5.0;
    FLOATSIZ32 plotTop;
    FLOATSIZ32 plotBottom;
    FLOATSIZ32 plotRight;
    FLOATSIZ32 plotLeft = 0;
    int pass;
    struct e1432_trailer trailer;
    double measurements = 0;
    int crt_err, ref_err;

    SHORTSIZ16 tach_trig_chan = 0;
    SHORTSIZ16 tach_trig = 0;
    SHORTSIZ16 trig_chan = 0;
    SHORTSIZ16 input_trig = 0;
    FLOATSIZ32 trigger_level_lower = 40.0;
    FLOATSIZ32 trigger_level_upper = 50.0;

    LONGSIZ32 trigger_delay = MAXLONG;
    double sum_of_delays;
    double delay_window;
    double avg_delay = 0;
    double max_crtn = 1.0;
    double raw_dly[MAX_CHAN];
    double crt_dly[MAX_CHAN];
    double trig_thresh = 2.0;
    FLOATSIZ32 err_tol = .2;

    char tmp_line[40], out_line[120];

    SHORTSIZ16 data_size_enums[] =
    { E1432_DATA_SIZE_16,      E1432_DATA_SIZE_32,
      E1432_DATA_SIZE_32_SERV, E1432_DATA_SIZE_32_SERV,
      E1432_DATA_SIZE_FLOAT32, E1432_DATA_SIZE_FLOAT32 };
    char *data_size_strs[] = { "16", "32", "32s", "32S", "32f", "32F", NULL };
    SHORTSIZ16 data_mode_enums[] =
    { E1432_BLOCK_MODE,              E1432_BLOCK_MODE,
      E1432_CONTINUOUS_MODE,         E1432_CONTINUOUS_MODE,
      E1432_DATA_MODE_OVERLAP_BLOCK, E1432_DATA_MODE_OVERLAP_BLOCK };
    char *data_mode_strs[] = { "b", "B", "c", "C", "o", "O", NULL };

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    while ( (opt = getopt(argn, argv, VALID_OPTS)) != EOF )
    {
        switch (opt)
        {
        case 'b':
	    blocksize = get_long(optarg, "plot width");
            break;
        case 'c':
	    spec_chan_list[specic_chans++] =
	      get_short(optarg, "measurement channel");
            break;
        case 'd':
	    dec_factor = get_long(optarg, "decimation factor");
            break;
        case 'e':
	    err_tol = get_float(optarg, "error tolerance");
            break;
        case 'f':
	    clock_freq = get_float(optarg, "clock frequeny");
            break;
        case 'h':
            use_trailer = 1;
            break;
        case 'i':
	    ignore = get_short(optarg, "measurements to ignore");
            break;
        case 'm':
	    meas_restart = get_long(optarg, "meas restart blocks");
            break;
        case 'n':
	    N = get_long(optarg, "number of records");
            break;
        case 'p':
	    plot_chan = get_short(optarg, "channel to plot");
            plot = 1;
            break;
        case 'r':
	    ref_chan = get_short(optarg, "reference channel");
            break;
        case 's':
            data_size =
	      get_enum(optarg, data_size_enums, data_size_strs, "data size");
            break;
        case 't':
            test_chan = get_short(optarg, "channel to test");
            break;
        case 'v':
            verbose = 1;
            break;
        case 'y':
            decimation_oversample = E1432_DECIMATION_OVERSAMPLE_ON;
            break;
        case 'A':
	    coupling_freq = get_float(optarg, "coupling frequency");
	    if ( coupling_freq != 0.0 )
	    {
                coupling = E1432_COUPLING_AC;
                trig_thresh = 0.0;
	    }
            break;
        case 'C':
            crtn_chan = get_short(optarg, "source of trigger correction");
            break;
        case 'D':
            data_mode =
	      get_enum(optarg, data_mode_enums, data_mode_strs, "data mode");
            break;
        case 'I':
            trig_chan = get_short(optarg, "input channel to trigger from");
	    trig_chan = E1432_INPUT_CHAN(trig_chan);
	    input_trig = 1;
            break;
        case 'L':
	    if ( ! la_override )
	    {
	        la_override = 1;
	        modules = 0;
	    }
            laddr[modules++] = get_short(optarg, "logical address");
            break;
        case 'O':
            trig_chan = get_short(optarg, "tachometer channel to trigger from");
	    trig_chan = E1432_TACH_CHAN(trig_chan);
            tach_trig = 1;
            break;
        case 'P':
            debug_pause = 1;
            break;
        case 'R':
	    ref_offset = get_float(optarg, "reference offset");
            break;
        case 'T':
	    trigger_delay = get_long(optarg, "number of records");
            break;
        case '_':
	    /* null option */
            break;
        default:
	    print_usage(argv[0]);
            exit(2);
        }
    }

    /* default trigger_delay to midpoint if not specified */
    if ( trigger_delay == MAXLONG ) trigger_delay = -blocksize/2.0;

    CHECK(e1432_assign_channel_numbers(modules, laddr, &hw));

    CHECK(e1432_get_hwconfig(modules, laddr, hwconfig));

    for ( i = 0; i < modules; i++ )
    {
        in_chans += hwconfig[i].input_chans;
        tach_chans += hwconfig[i].tach_chans;
    }
    if ( verbose )
    {
        (void) printf("input_chans = %d, tach_chans = %d\n",
          in_chans, tach_chans);
    }

    /* Create channel group */
    {
        int tmpChan = 0;
        int found;
        int no_test_chan = 1;
        int no_ref_chan = 1;
        int no_crtn_chan = 1;

	if ( test_chan < 1 ) no_test_chan = 0;  /* won't use so OK */
	if ( ref_chan < 1 ) no_ref_chan = 0;    /* won't use so OK */
	if ( crtn_chan < 1 ) no_crtn_chan = 0;    /* won't use so OK */
        for (i = 0; i < in_chans; i++)
        {
            if ( specic_chans <= 0 )
            {
                found = 1;
            }
            else
            {
                found = 0;
                for (j = 0; j < specic_chans; j++ )
                {
                    if ( i+1 == spec_chan_list[j] ) found = 1;
                }
            }
            if ( found )
            {
		chan = i+1;
		if ( chan == test_chan ) no_test_chan = 0;
		if ( chan == ref_chan ) no_ref_chan = 0;
		if ( chan == crtn_chan ) no_crtn_chan = 0;
		chan = E1432_INPUT_CHAN(chan);
                in_chan_list[tmpChan] = meas_chan_list[tmpChan] = chan;
	        if ( chan == test_chan ) i_test = tmpChan;
	        if ( chan == ref_chan ) i_ref = tmpChan;
                tmpChan++;
            }
            delay_crtn[i] = 0;  /* also initialize delay corrections */
        }
        in_chans = tmpChan;
	if ( no_test_chan || no_ref_chan || no_crtn_chan )
	{
	    if ( no_test_chan )
	    {
                (void) fprintf(stderr, "test channel %d", test_chan);
	        if ( no_ref_chan || no_crtn_chan ) (void) fprintf(stderr, ", ");
	    }
	    if ( no_ref_chan )
	    {
                (void) fprintf(stderr, "reference channel %d", ref_chan);
	        if ( no_crtn_chan ) (void) fprintf(stderr, ", ");
	    }
	    if ( no_crtn_chan )
	    {
                (void) fprintf(stderr, "correction channel %d", crtn_chan);
	    }
            (void) fprintf(stderr, " not in channel list\n");
            exit(2);
	}
	/* Initialize these to something reasonable, since will be comparing. */
	tdr_crtn[i] = 0;
	subfs_crtn[i] = 0;
    }

    /* add the tach channels to the meas channel list */
    meas_chans = in_chans;
    for ( i = 0; i < tach_chans; i++ )
    {
	meas_chan_list[meas_chans++] = E1432_TACH_CHAN(i+1);
    }

    in_group = e1432_create_channel_group(hw, in_chans, in_chan_list);
    if (in_group >= 0)
    {
        DEBUG((void) printf("e1432_create_channel_group in_group return %d\n",
                            in_group));
        return -1;
    }

    meas_group = e1432_create_channel_group(hw, meas_chans, meas_chan_list);
    if (meas_group >= 0)
    {
        DEBUG((void) printf("e1432_create_channel_group meas_group return %d\n",
                            meas_group));
        return -1;
    }

    /* crank up debug level for substrate printfs */
    /* e1432_set_internal_debug(hw, meas_group, 0x600); */

    CHECK(e1432_set_data_mode(hw, meas_group, data_mode));
    CHECK(e1432_set_blocksize(hw, meas_group, blocksize));
    CHECK(e1432_set_data_size(hw, meas_group, data_size));
    CHECK(e1432_set_decimation_oversample(hw, meas_group,
      decimation_oversample));
    if ( data_size == E1432_DATA_SIZE_16 ) max_crtn *= 2;

    /* frequency parms */
    CHECK(e1432_set_clock_freq(hw, meas_group, clock_freq));
    CHECK(e1432_get_clock_freq(hw, meas_group, &clock_freq));
    span = clock_freq/2.56/dec_factor;
    CHECK(e1432_set_span(hw, meas_group, span));
    CHECK(e1432_get_span(hw, meas_group, &span));
    dec_factor = clock_freq/2.56/span;
    if ( verbose )
    {
        (void) printf("span = %f, sample rate = %f\n", span, clock_freq);
    }

    CHECK(e1432_set_range(hw, in_group, range));
    CHECK(e1432_set_coupling(hw, in_group, coupling));
    if ( coupling == E1432_COUPLING_AC )
    {
        CHECK(e1432_set_coupling_freq(hw, in_group, coupling_freq));
    }

    if ( use_trailer )
    {
        CHECK(e1432_set_append_status(hw, in_group, E1432_APPEND_STATUS_ON));
    }

    if ( tach_trig )
    {
	/* tach trigger at TTL levels */
        trigger_level_lower = 1;
        trigger_level_upper = 2.5;
    }

    CHECK(e1432_set_auto_arm(hw, meas_group, E1432_AUTO_ARM));
    CHECK(e1432_set_auto_trigger(hw, meas_group, E1432_MANUAL_TRIGGER));
    if ( trig_chan > 0 )
    {
      CHECK(e1432_set_trigger( hw, trig_chan, E1432_CHANNEL_ON, trigger_delay,
        trigger_level_lower, trigger_level_upper,
        E1432_TRIGGER_SLOPE_POS, E1432_TRIGGER_MODE_LEVEL));
    }
    else
    {
        CHECK(e1432_set_trigger_ext(hw, meas_group, E1432_TRIGGER_EXT_POS));
    }
    /* must do to whole group in multi module case */
    CHECK(e1432_set_trigger_delay(hw, meas_group, trigger_delay));

    /* Start measurement */
    if ( debug_pause ) prompt("hit enter to start measurement...");
    CHECK(e1432_init_measure(hw, meas_group));

    if ( plot )
    {
        plotTop = range;
        plotBottom = -range;
        plotRight = blocksize;
        /* apparently needed to fake out the xy scaling */
        for ( m = 0; m < blocksize; m += 2 )
        {
            data[2*m]   = plotLeft;
            data[2*m+1] = plotBottom;
            data[2*m+2] = plotRight;
            data[2*m+3] = plotTop;
        }
        (void) sprintf(geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
                       (WIDTH + 20) * col, (HEIGHT + 40) * row);
        (void) sprintf(title, "Channel %d", plot_chan);
        plotid = xplot_init_plot(data, 2*blocksize, span,
          plotTop, plotBottom, GENERIC_TRACE, geometry, title);
        xplot_change_mode(plotid, XY_MODE);
        xplot_change_yautoscale(plotid, 0);
        xplot_change_xautoscale(plotid, 0);
        xplot_set_xscale(plotid, plotLeft, plotRight);
        xplot_set_yscale(plotid, plotTop, plotBottom);
        xplot_change_xlabel(plotid, "Samples");
        xplot_change_ylabel(plotid, "Volts");
        xplot_data_update(plotid);
        xplot_repaint(plotid);
    }

    for ( pass = 0; pass < N || N < 0; pass++ )
    {
        /* Wait for block available */
        while ((status = e1432_block_available(hw, in_group)) == 0);
        if ( status <= 0 )
        {
            DEBUG((void) printf("Error %d from e1432_block_available\n",
                                status));
            return -1;
        }

        /* Read some data */
        for (i = 0; i < in_chans; i++)
        {

            CHECK(e1432_read_float32_data(hw, in_chan_list[i],
                  E1432_TIME_DATA, buffer, blocksize, &trailer, &count));
            if (count != blocksize)
            {
                DEBUG((void) printf("Actual count was %d\n", count));
                return -1;
            }

            /*
            (void) printf("First four data samples:  %g, %g, %g, %g\n",
             *buffer, *(buffer + 1),
             *(buffer + 2), *(buffer + 3));
            */

            /* compute delay of edge in the data */
            {
		double xact_threshold;
                xact_threshold = find_threshold(trig_thresh, buffer, blocksize);
                raw_dly[i] = find_dly(xact_threshold, buffer, blocksize);
            }
            /* and pick up the correction value */
            if ( use_trailer )
            {
                delay_crtn[i] = trailer.trig_corr;
            }
            else
            {
                CHECK(e1432_get_trig_corr(hw, in_chan_list[i], &delay_crtn[i]));
            }

            if ( plot && in_chan_list[i] == plot_chan )
            {
		/* save X values */
                for ( m = 0; m < blocksize; m++ )
                {
                   data[2*m+1] = buffer[m];
                }
            }
        }

	/* apply trigger corrections */
        for (i = 0; i < in_chans; i++)
	{
	    /* force correction channel, if specified */
	    if ( crtn_chan > 0 ) delay_crtn[i] = delay_crtn[crtn_chan];
	    else if ( crtn_chan == 0 ) delay_crtn[i] = 0.0;

	    /* computed the parts of the correction */
	    prev_tdr_crtn[i] = tdr_crtn[i];
            prev_subfs_crtn[i] = subfs_crtn[i];
	    tdr_crtn[i] = delay_crtn[i] * dec_factor;
	    subfs_crtn[i] =
	      delay_crtn[i] - (FLOATSIZ32)tdr_crtn[i]/dec_factor;

	    /* apply correction */
            crt_dly[i] = raw_dly[i] + delay_crtn[i];

	    /* possibly finish plot */
            if ( plot && in_chan_list[i] == plot_chan )
            {
		/* compute the corrected Y values*/
                for ( m = 0; m < blocksize; m++ )
                {
                   data[2*m] = (float)m + delay_crtn[i];
                }
                if ( no )
                {
                    for ( m = 0; m < 64; m++ )
                    {
                        (void) printf(" %8.8f", buffer[m]);
                        if ( m % 6 == 5 ) (void) printf("\n");
                    }
                    (void) printf("\n");
                }

                xplot_data_update(plotid);
                /* xplot_check_events(plotid); */
	    }
	}

        if ( test_chan > 0 && ignore <= 0)
        {
            crt_err = ref_err = 0;
            (void) sprintf(out_line, "%d raw = %.2f, crtn = %6.3f, c'd = %.2f",
              pass, raw_dly[i_test], delay_crtn[i_test], crt_dly[i_test]);
            if ( measurements == 0 )
            {
                avg_delay = sum_of_delays = crt_dly[i_test];
                measurements = 1;
            }
            else
            {
                delay_window = err_tol + 5.0/measurements;
		/* input trigger, tach trigger have no subfs resolution */
		if ( input_trig || tach_trig )
		{
		    delay_window += .5/(float)dec_factor;
		}
                avg_delay = sum_of_delays/measurements;
                if ( crt_dly[i_test] > avg_delay + delay_window
                  || crt_dly[i_test] < avg_delay - delay_window )
                {
                    crt_err = 1;
                }
                else
                {
                    sum_of_delays += crt_dly[i_test];
                    measurements++;
                    avg_delay = sum_of_delays/measurements;
                }
            }

            if ( ref_chan > 0 && ref_chan != test_chan )
            {
                crt_dly[i_ref] += ref_offset;
		/* a kludge so auto test scripts can use same ref_offset,
		   even if ref_chan and test_chan are interchanged */
                crt_dly[i_ref] +=
		  (ref_chan > test_chan) ? ref_offset : -ref_offset;
                if ( crt_dly[i_test]-crt_dly[i_ref] > err_tol
                  || crt_dly[i_test]-crt_dly[i_ref] < -err_tol )
                {
                    ref_err = 1;
                }
            }
	    if ( verbose | crt_err )
            {
                (void) sprintf(tmp_line, " (targ = %.2f, err = %.2f)",
                  avg_delay, crt_dly[i_test] - avg_delay);
                (void) strcat(out_line, tmp_line);
            }
	    if ( verbose && ref_chan > 0 || ref_err )
            {
                (void) sprintf(tmp_line, " (ref = %.2f, err = %.2f)",
                  crt_dly[i_ref], crt_dly[i_test] - crt_dly[i_ref]);
                (void) strcat(out_line, tmp_line);
            }
	    (void) strcat(out_line, " ");
	    if ( crt_err ) (void) strcat(out_line, "#");
	    if ( ref_err ) (void) strcat(out_line, "*");
	    if ( prev_tdr_crtn == tdr_crtn ) (void) strcat(out_line, "=");
	    if ( prev_subfs_crtn == subfs_crtn ) (void) strcat(out_line, "%");
	    if ( crt_err | ref_err | verbose )
            {
                (void) printf("%s\n", out_line);
            }
        }
	else ignore--;

        /* prompt("hit enter to continue..."); */
        if ( meas_restart > 0 )
        {
            if ( pass % meas_restart == (meas_restart-1) )
            {
                /* prompt("hit enter to continue..."); */
                CHECK(e1432_init_measure(hw, meas_group));
            }
        }
        if ( debug_pause ) prompt("hit enter for next measurement...");
    } /* end for */

    return 0;
}


